function [ ] = migration( searchPattern )
%MIGRATION Tool to migrate CageViewer result files to current version
%   Call the script by specifying the file pattern (e.g 'data/*.mat')
%   All files will be OVERWRITTEN!
%   Example usage:
%   >> migration('data/*.mat')
    VERSION = 0.92;
    
    fprintf('Protocol migration %1.2f\n', VERSION);
    files = ls(searchPattern);
    dir = fileparts(searchPattern);
    if isempty(dir)
        dir = '.';
    end
    [nFiles, t1] = size(files);
    for i = 1:nFiles
        file = files(i, :);
        file = [dir filesep file];
        fprintf('Processing file %s ... ', file);
        proto = load(file);
        try
            %If NPoints is present this is an old file
            proto.NPoints;
            fprintf('detected a Format < V1.0 protocol format!\n')
            migrate_old(file, false)
        catch            
            try
                %There was a tipo, test for that
                if proto.applicatinVersion < 0.98
                    fprintf('detected Format V1.0 protocol format!\n')
                    migrate_v1(file, false)
                else
                    fprintf('new protocl format, no migration needed!\n')
                end            
            catch
                fprintf('new protocl format, no migration needed!\n')
            end            
        end
    end

end

function migrate_old(filename, debug)
try
    old = load(filename);
    new = {};
    
    new.application = 'CageViewer';
    new.applicationVersion = 0.98;
    new.sensorDataMeta = {'Time'  'HouseLight'  'BlueCueLight'  'YellowCueLight'  'Shock'  'Valve'  'Sound'  'Lick'  'NosePoke'};
    new.periodsExecutedMeta = {'TrialCnter'  'TrialIdx'  'PeriodIdx'  'onSetIdx'  'offSetIdx'};
    new.startDate = '';
    new.endDate = '';
    new.protocolFileName = [old.ProtocolDef.FilePath old.ProtocolDef.Filename];
    new.nLoopIterations = old.NPoints-1; 
    
    new.protocolMeta = {};
    new.protocolMeta.seed = '';
    new.protocolMeta.group = '';
    new.protocolMeta.animalNumber = '';
    new.protocolMeta.phase = '';
    new.protocolMeta.session = '';
    new.protocolMeta.videoMode = 'UNKNOWN';
    
    %Join Sensor Data
    new.sensorData = [double(old.Time); double(old.HouseLight); double(old.CueLight); double(old.CueLight); double(old.Shock); double(old.RewardDelivery); double(old.Sound); 1.0 - double(old.RewardIR); double(old.NosePoke)]';
    new.sensorData = new.sensorData(1:new.nLoopIterations, :);
    
    %%%% Reconstruct in part protocol
    new.protocol = {};
    new.protocol.protocolVersion = 4.3;
    new.protocol.maxProtocolDuration = old.ProtocolDef.LimitDuration;
    new.protocol.finishLastTrial = 0;
    new.protocol.TrialsDefinition = cell(1, old.TrialsAvail.N);
    
    new.protocol.Sounds = cell(1, 0);
    for i = 1:old.TrialsAvail.N
        new.protocol.TrialsDefinition{i} = {};
        new.protocol.TrialsDefinition{i}.name = old.TrialsAvail.Names{i};
        new.protocol.TrialsDefinition{i}.periods = old.TrialsAvail.Trials{i};
        for j = 1:length(new.protocol.TrialsDefinition{i}.periods)
            if ~ strcmp(new.protocol.TrialsDefinition{i}.periods{j}.SoundFile, 'No sound file chosen')
                new.protocol.Sounds{1, end+1} = new.protocol.TrialsDefinition{i}.periods{j}.SoundFile;
            end
            new.protocol.TrialsDefinition{i}.periods{j}.name = new.protocol.TrialsDefinition{i}.periods{j}.Name;
        end
    end
    new.protocol.Sounds = unique(new.protocol.Sounds);
    
    new.protocol.TrialSequence = cell(1,0);
    for i = 1:length(old.ProtocolDef.Times)
        times = old.ProtocolDef.Times{i};
        new.protocol.TrialSequence{1, end+1} = repmat(old.ProtocolDef.TrialsIdxAssembly{i}, 1, times);        
    end
    
    %%% Reconstruct periodsExecuted
    trialCnt = 0;
    new.periodsExecuted = cell(0, 5);
    for i = 1:length(old.TrialPeriodActuallyExecutedID)
        trialCnt = trialCnt + 1;
        trial = new.protocol.TrialSequence{i};                
        periods = old.TrialPeriodActuallyExecutedID{i};
        for j = 1:length(periods)
            [r, c] = size(new.periodsExecuted);
            r = r + 1;
            new.periodsExecuted{r, 1} = trialCnt;
            new.periodsExecuted{r, 2} = trial;
            new.periodsExecuted{r, 3} = periods{j};
            new.periodsExecuted{r, 4} = old.TrialTimeIndices{i}{periods{j}} - 1;
            new.periodsExecuted{r, 5} = 0; %Fill later
        end        
    end
        
    [r, c] = size(new.periodsExecuted);
    for i = 1:(r-1)
        new.periodsExecuted{i, 5} = new.periodsExecuted{i+1, 4} - 1;
    end
    new.periodsExecuted{end, 5} = new.nLoopIterations;
    new.periodsExecuted = cell2mat(new.periodsExecuted);
    %%% 

    %Make the field order the same between files
    new = orderfields(new);  
    if debug
        [a, b, c] = fileparts(filename);
        save([a filesep 'migration_' b c], '-struct', 'new')
    else
        save(filename, '-struct', 'new')   
    end    
    catch err
        warning('Migrating file %s failed! %s (%d) : %s', filename, err.stack(1).file, err.stack(1).line, err.message)
end
end

function migrate_v1(filename, debug)
try
    old = load(filename);
    new = {};
        
    new.application = 'CageViewer';
    new.applicationVersion = 0.98;
    new.sensorDataMeta = {'Time'  'HouseLight'  'BlueCueLight'  'YellowCueLight'  'Shock'  'Valve'  'Sound'  'Lick'  'NosePoke'};
    new.periodsExecutedMeta = {'TrialCnter'  'TrialIdx'  'PeriodIdx'  'onSetIdx'  'offSetIdx'};    
    new.startDate = '';
    new.endDate = '';    
    new.nLoopIterations = length(old.sensorData);
    %Add protocolFileName field
    new.protocolFileName = 'UNKNOWN';    
    
    %Store protocol as structe not matlab object
    new.protocol = struct(old.protocol);      
    
    %Add Sensor Data
    new.sensorData = old.sensorData;
    
    %Remove volume field
    new.protocolMeta = rmfield(old.protocolMeta, 'volume');
    
    %Add Trial Counter to periodsExecuted
    [r, c] = size(old.periodsExecuted);
    if c < 5
        cnt = 1;
        N = length(old.periodsExecuted);
        trialCnt = ones(N, 1);
        nTrials = sum(old.periodsExecuted(:,2) == 1);
        for nT = 1:nTrials
            trialCnt(cnt) = nT;
            cnt = cnt + 1;            
            while(old.periodsExecuted(cnt, 2) ~= 1 && cnt < N)
                trialCnt(cnt) = nT;            
                cnt = cnt + 1;            
            end                            
        end
        trialCnt(cnt) = nT;
        new.periodsExecuted = horzcat(trialCnt, old.periodsExecuted);    
    else
        new.periodsExecuted = old.periodsExecuted;
        
        %Check for a error where the endIndex was NaN
        if isnan(new.periodsExecuted(1, 5))
            new.periodsExecuted(:, 5) = [new.periodsExecuted(2:end, 4); new.nLoopIterations];
        end
    end
        
    %Make the field order the same between files
    new = orderfields(new);
    if debug
        [a, b, c] = fileparts(filename);
        save([a filesep 'migration_' b c], '-struct', 'new')
    else
        save(filename, '-struct', 'new')   
    end
catch err
    warning('Migrating file %s failed! %s', filename, err.message)
end
end

